#include <stdio.h>
#include <stdlib.h>

#include <sys/time.h>
#include <sys/resource.h>

#include <cilk/cilk.h>
#include <cilk/cilk_api.h>

int l;

int sequential_fib (int n) {
  if (n < 2) return n;
  int x = sequential_fib (n - 1);
  int y = sequential_fib (n - 2);
  return x + y;
}

int parallel_fib (int n) {
  if (n < l) return sequential_fib (n);
  int x = cilk_spawn parallel_fib (n - 1);
#if 0
  int y = cilk_spawn parallel_fib (n - 2);
#else
  int y = parallel_fib (n - 2);
#endif
#if 1
  cilk_sync;
  int r = x + y;
  return r;
#else
  return x + y;
#endif
}

double wall_clock_time () {
  double res = 0;
  struct timeval tv;
  if (!gettimeofday (&tv, 0))
  res = 1e-6 * tv.tv_usec, res += tv.tv_sec;
  return res;
}

double process_time () {
  struct rusage u;
  if (getrusage (RUSAGE_SELF, &u)) return 0;
  double res = u.ru_utime.tv_sec + 1e-6 * u.ru_utime.tv_usec;
  res += u.ru_stime.tv_sec + 1e-6 * u.ru_stime.tv_usec;
  return res;
}

int main (int argc, char ** argv) {
  if (argc < 2 || argc > 4) {
    printf (
"usage: cfib <n> [ <w> [ <l> ] ]\n"
"\n"
"Computes fibonacci number <n>\n"
"using <w> worker threads (default 1) and\n"
"leaf coarsening limit <l> (default 2).\n");
    exit (0);
  }
  int n = atoi (argv[1]);
  const char * warg = argc > 2 ? argv[2] : "1";
  if (argc > 3) l = atoi (argv[3]);
  if (l < 2) l = 2;
  int w = atoi (warg);
  __cilkrts_end_cilk ();  
  __cilkrts_set_param ("nworkers", warg); 
  double s = wall_clock_time ();
  int f = parallel_fib (n);
  double t = wall_clock_time () - s;
  double p = process_time ();
  double u = (t && w) ? 100.0 * p / t / (double) w : 0;
  printf (
    "fib (%d) = %d with %d threads leaf limit %d in %.2f seconds\n"
    "%.2f process time, utilization %.0f%%\n",
    n, f, w, l, t, p, u);
  return 0;
}
