#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/wait.h>
#include <sys/sem.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

#define NUM_CHILDREN 5
#define SLEEP_LIMIT 3

/* this example uses System V semaphores and System V shared memory */

#define ENABLE_SEMAPHORE

static int *shared_sum_ptr;
static int semid;

static struct sembuf lock_semaphore = { 0, -1, 0 };
static struct sembuf unlock_semaphore = { 0, 1, 0 };

static int
inc (void)
{
  int sum, return_val;
  srand (getpid ());
  return_val = 0;
  sleep (rand () % (SLEEP_LIMIT + 1));
#ifdef ENABLE_SEMAPHORE
  if (semop (semid, &lock_semaphore, 1) == -1)
    {
      printf ("semop failed");
      return_val = 1;
    }
#endif
  sum = *shared_sum_ptr;
  sleep (rand () % (SLEEP_LIMIT + 1));
  printf ("Process %d (parent %d) increments value\n", getpid (), getppid ());
  fflush (stdout);
  *shared_sum_ptr = sum + 1;
#ifdef ENABLE_SEMAPHORE
  if (semop (semid, &unlock_semaphore, 1) == -1)
    {
      printf ("semop failed");
      return_val = 1;
    }
#endif
  return return_val;
}

int
main (void)
{
  pid_t children[NUM_CHILDREN];
  pid_t pid;
  int status, i, error, shared_mem_id;
  struct shmid_ds shmid_ds_struct;

  error = 0;

  /* get shared memory id for sum */
  shared_mem_id = shmget (IPC_PRIVATE, sizeof (int), 0600);
  if (shared_mem_id == -1)
    {
      printf ("shmget failed\n");
      exit (EXIT_FAILURE);
    }
  /* attach memory to process */
  shared_sum_ptr = (int *) shmat (shared_mem_id, NULL, 0);
  /* shmat returns -1 on error */
  if (shared_sum_ptr == (void *) -1l)
    {
      printf ("shmat failed\n");
      shmctl (shared_mem_id, IPC_RMID, &shmid_ds_struct);
      exit (EXIT_FAILURE);
    }

  /* init sum */
  *shared_sum_ptr = 0;

  /* get semaphore id */
  semid = semget (IPC_PRIVATE, 1, 0600);
  if (semid == -1)
    {
      printf ("semget failed\n");
      shmctl (shared_mem_id, IPC_RMID, &shmid_ds_struct);
      exit (EXIT_FAILURE);
    }

  /* init semaphore */
  if (semctl (semid, 0, SETVAL, 1) == -1)
    {
      printf ("semctl failed\n");
      semctl (semid, 0, IPC_RMID);
      shmctl (shared_mem_id, IPC_RMID, &shmid_ds_struct);
      exit (EXIT_FAILURE);
    }

  for (i = 0; i < NUM_CHILDREN; i++)
    {
      if ((pid = fork ()) == 0)
        {
          /* only child enters */
          if (inc ())
            exit (EXIT_FAILURE);
          exit (EXIT_SUCCESS);
        }
      else
        {
          /* only parent enters */
          /* error ? */
          if (pid == -1)
            {
              printf ("fork failed\n");
              semctl (semid, 0, IPC_RMID);
              shmctl (shared_mem_id, IPC_RMID, &shmid_ds_struct);
              exit (EXIT_FAILURE);
            }
          assert (pid > 0);
          /* remember pid of child */
          children[i] = pid;
        }
    }

  /* wait for children */
  for (i = 0; i < NUM_CHILDREN; i++)
    {
      if (waitpid (children[i], &status, 0) != children[i])
        error = 1;
      else
        {
          if (!WIFEXITED (status) || WEXITSTATUS (status) != EXIT_SUCCESS)
            error = 1;
        }
    }
  printf ("Result: %d\n", *shared_sum_ptr);
  /* clean up */
  if (semctl (semid, 0, IPC_RMID) == -1)
    {
      printf ("semctl failed\n");
      error = 1;
    }
  if (shmctl (shared_mem_id, IPC_RMID, &shmid_ds_struct) == -1)
    {
      printf ("shmctl failed\n");
      error = 1;
    }
  printf ("Main process %d finishes\n", getpid ());
  if (error)
    exit (EXIT_FAILURE);
  exit (EXIT_SUCCESS);
}
