// Copyright (c) 2020 Armin Biere Johannes Kepler University Linz

#include <assert.h>
#include <ctype.h>
#include <limits.h>
#include <pthread.h>
#include <stdarg.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/resource.h>
#include <sys/time.h>
#include <unistd.h>

/*------------------------------------------------------------------------*/

// Messages.

static void
error_prefix (const char *type)
{
  if (isatty (2))
    fprintf (stderr, "\033[1mparfail:\033[1;31m %s:\033[0m ", type);
  else
    fprintf (stderr, "parfail: %s: ", type);
}

static void
die (const char *fmt, ...)
{
  va_list ap;
  error_prefix ("error");
  va_start (ap, fmt);
  vfprintf (stderr, fmt, ap);
  va_end (ap);
  fputc ('\n', stderr);
  exit (1);
}

static pthread_mutex_t message_mutex = PTHREAD_MUTEX_INITIALIZER;

static void
msg (const char *fmt, ...)
{
  if (pthread_mutex_lock (&message_mutex))
    die ("failed to lock messages");
  va_list ap;
  fputs ("c [parfail] ", stdout);
  va_start (ap, fmt);
  vprintf (fmt, ap);
  va_end (ap);
  fputc ('\n', stdout);
  fflush (stdout);
  if (pthread_mutex_unlock (&message_mutex))
    die ("failed to unlock messages");
}

/*------------------------------------------------------------------------*/

// Formula.

static int num_clauses;
static int num_variables;

static int **clauses;		// All clauses (zero terminated).
static int ***watches;		// Clauses watched by literal.

static int num_units;		// Number of original unit clauses.
static int *units;		// The original units found in the formula.

// Clauses are pointers to zero terminated arrays of literals. The 'watches'
// array then maps literals to a zero terminated array of pointers to
// clauses in which that literal occurs.  Note that variables are in the
// range '1..<num_variables>' and literals either variables or their
// negation. After allocating 'watches' it needs to be shifted such that it
// can be accessed by negative numbers too.  Similarly negatively indexed
// arrays are 'values' and 'stamps' (as well as 'count' below).

static void
init_watches (void)
{
  unsigned *count = calloc (2 * (num_variables + 1u), sizeof *count);
  if (!count)
    die ("failed to allocated counter");
  count += num_variables;

  // First count number of occurrences of all literals.
  //
  for (int i = 0; i < num_clauses; i++)
    for (const int *p = clauses[i]; *p; p++)
      count[*p]++;

  watches = malloc (2 * (num_variables + 1u) * sizeof *watches);
  if (!watches)
    die ("failed to allocated watches");
  watches += num_variables;

  // Now allocate watcher arrays of appropriate size for each variable.
  //
  for (int lit = -num_variables; lit <= num_variables; lit++)
    if (!(watches[lit] = malloc ((count[lit] + 1u) * sizeof (int *))))
      die ("failed to allocate watches[%d]", lit);

  // Insert the clauses into the watcher arrays of each literal.
  //
  for (int i = 0; i < num_clauses; i++)
    for (const int *p = clauses[i]; *p; p++)
      *watches[*p]++ = clauses[i];

  // Add terminating zero and reset pointers back to start.
  //
  for (int lit = -num_variables; lit <= num_variables; lit++)
    *watches[lit] = 0, watches[lit] -= count[lit];

  count -= num_variables;
  free (count);
}

static void
reset_watches (void)
{
  for (int lit = -num_variables; lit <= num_variables; lit++)
    free (watches[lit]);

  watches -= num_variables;
  free (watches);
}

static void
reset_clauses (void)
{
  for (int i = 0; i < num_clauses; i++)
    free (clauses[i]);
  free (clauses);

  free (units);
}

/*------------------------------------------------------------------------*/

// Global result.

// The 'inconsistent' flag is read and written without locking.

static volatile bool inconsistent;	// Formula unsatisfiable.

// The 'learned' stack with top 'num_learned' is allocated before the worker
// threads starts.  Learned units (negation of failed literals) are pushed on
// it, but not all implied literals.  The workers test whether there are new
// learned literals and if so need to restart a new probing round. Thus the
// worker threads mainly communicate over this stack.

static int num_learned;		// Number of learned (unit) clauses.
static int *learned;		// Stack of all learned (unit) clauses.

// There is only one lock to access 'num_learned', 'learned' and 'completed'.
// The 'completed' field of each worker is used to determine whether the
// corresponding worker has already propagated all the learned literals.
// If all workers have completed propagation of all learned literals then the
// workers terminate.  This is tested by each worker individually.  For all
// this synchronization we only use the following single lock.

static pthread_mutex_t learned_mutex = PTHREAD_MUTEX_INITIALIZER;

static void
init_learned_unit_clauses (void)
{
  learned = calloc (num_variables, sizeof *learned);
  if (num_variables && !learned)
    die ("failed to allocate learned unit clauses");
}

/*------------------------------------------------------------------------*/

// Needed for printing and statistics at the end.  These implied literals
// are set by the simplification worker, which should have a consistent
// global assignment.  The learned units above are a subset.

static int num_implied;		// Total number of implied literals.
static int *implied;		// Stack of these implied literals.

/*------------------------------------------------------------------------*/

// Parsing.

static const char *input_path;
static FILE *input_file;
static int close_input;		// 1=fclose, 2=pclose

static bool
is_number (const char *arg)
{
  if (!*arg)
    return false;
  for (const char *p = arg; *p; p++)
    if (!isdigit (*p))
      return false;
  return true;
}

static long lineno = 1;

static int
read_next_char (void)
{
  int res = getc (input_file);
  if (res == '\n')
    lineno++;
  return res;
}

static void
parse_error (const char *fmt, ...)
{
  va_list ap;
  error_prefix ("parse error");
  fprintf (stderr, " in '%s' line %ld: ", input_path, lineno);
  va_start (ap, fmt);
  vfprintf (stderr, fmt, ap);
  va_end (ap);
  fputc ('\n', stderr);
  exit (1);
}

static void
parse (void)
{
  int ch;
  for (;;)
    {
      ch = read_next_char ();
      if (ch == 'c')
	{
	  while ((ch = read_next_char ()) != '\n')
	    if (ch == EOF)
	      parse_error ("end-of-file in comment");
	}
      else if (ch == 'p')
	break;
      else
	parse_error ("expected 'p' or 'c'");
    }

  if (read_next_char () != ' ' ||
      read_next_char () != 'c' ||
      read_next_char () != 'n' ||
      read_next_char () != 'f' || read_next_char () != ' ')
    parse_error ("expected 'p cnf '");
  if (!isdigit (ch = read_next_char ()))
    parse_error ("expected digit after 'p cnf '");
  num_variables = ch - '0';
  while (isdigit (ch = read_next_char ()))
    {
      if (INT_MAX / 10 < num_variables)
	parse_error ("number of variables too large");
      num_variables *= 10;
      const int digit = ch - '0';
      if (INT_MAX - digit < num_variables)
	parse_error ("number of variables too large");
      num_variables += digit;
    }
  if (ch != ' ')
    parse_error ("expected space after 'p cnf %d'", num_variables);
  if (!isdigit (ch = read_next_char ()))
    parse_error ("expected digit after 'p cnf %d '");
  num_clauses = ch - '0';
  while (isdigit (ch = read_next_char ()))
    {
      if (INT_MAX / 10 < num_clauses)
	parse_error ("number of clauses too large");
      num_clauses *= 10;
      const int digit = ch - '0';
      if (INT_MAX - digit < num_clauses)
	parse_error ("number of clauses too large");
      num_clauses += digit;
    }
  if (ch != '\n')
    parse_error ("expected new line after 'p cnf %d %d'",
		 num_variables, num_clauses);
  msg ("found 'p cnf %d %d' header", num_variables, num_clauses);

  clauses = malloc (num_clauses * sizeof *clauses);
  if (num_clauses && !clauses)
    die ("failed to allocate clauses");

  units = malloc (num_clauses * sizeof *units);
  if (num_clauses && !units)
    die ("failed to allocate original units");

  int *clause = 0;
  size_t allocated = 0;
  size_t size = 0;

  int parsed = 0;
  int lit = 0;

  for (;;)
    {
      ch = read_next_char ();
      if (ch == ' ' || ch == '\t' || ch == '\n')
	continue;
      if (ch == 'c')
	{
	  while ((ch = read_next_char ()) != '\n')
	    if (ch == EOF)
	      parse_error ("end-of-file in comment");
	  continue;
	}
      if (ch == EOF)
	{
	  if (parsed < num_clauses)
	    parse_error ("clause missing");
	  if (lit)
	    parse_error ("terminating zero missing");
	  break;
	}
      int sign = 1;
      if (ch == '-')
	{
	  sign = -1;
	  ch = read_next_char ();
	  if (!isdigit (ch))
	    parse_error ("expected digit after '-'");
	}
      else if (!isdigit (ch))
	parse_error ("expected literal");
      if (parsed == num_clauses)
	parse_error ("too many clauses");
      lit = ch - '0';
      while (isdigit (ch = read_next_char ()))
	{
	  if (INT_MAX / 10 < lit)
	    parse_error ("literal too large");
	  lit *= 10;
	  const int digit = ch - '0';
	  if (INT_MAX - digit < lit)
	    parse_error ("literal too large");
	  lit += digit;
	}
      if (lit > num_variables)
	parse_error ("literal too large");
      lit *= sign;
      assert (lit <= num_variables);
      if (ch != ' ' && ch != '\t' && ch != '\n')
	parse_error ("expected space after '%d'", lit);
      if (lit)
	{
	  if (size == allocated)
	    {
	      allocated = allocated ? 2 * allocated : 1;
	      clause = realloc (clause, allocated * sizeof *clause);
	      if (!clause)
		die ("failed to reallocate clause");
	    }
	  clause[size++] = lit;
	}
      else
	{
	  if (!(clauses[parsed] = malloc ((size + 1) * sizeof (int))))
	    die ("failed allocate clause");
	  memcpy (clauses[parsed], clause, size * sizeof (int));
	  clauses[parsed][size] = 0;
	  if (!size)
	    inconsistent = true;
	  if (size == 1)
	    units[num_units++] = clause[0];
	  parsed++;
	  size = 0;
	}
    }
  free (clause);

  msg ("found %d original unit clauses", num_units);
}

/*------------------------------------------------------------------------*/

// Threading.

#define ACCESS(V) (* (volatile typeof (V) *)  &(V))

static int num_threads;

static void
init_threads (void)
{
  if (num_threads)
    msg ("using %d threads as specified", num_threads);
  else if ((num_threads = sysconf (_SC_NPROCESSORS_ONLN)) > 0)
    msg ("using %d threads as determined by 'sysconf'", num_threads);
  else
    {
      num_threads = 1;
      msg ("using 1 thread (since 'sysconf' failed)");
    }
}

/*------------------------------------------------------------------------*/

// Statistics.

static double started;

static double
wall_clock_time (void)
{
  double res = 0;
  struct timeval tv;
  if (!gettimeofday (&tv, 0))
    res = 1e-6 * tv.tv_usec, res += tv.tv_sec;
  return res;
}

static double
process_time (void)
{
  struct rusage u;
  if (getrusage (RUSAGE_SELF, &u))
    return 0;
  double res = u.ru_utime.tv_sec + 1e-6 * u.ru_utime.tv_usec;
  res += u.ru_stime.tv_sec + 1e-6 * u.ru_stime.tv_usec;
  return res;
}

static long propagations;
static long decisions;
static long conflicts;

static double
average (double a, double b)
{
  return b ? a / b : 0;
}

static double
percent (double a, double b)
{
  return average (100 * a, b);
}

static void
statistics (void)
{
  double w = wall_clock_time () - started;
  double p = process_time ();

  msg ("");
  msg ("propagations %ld (%.2f millions per second)",
       propagations, average (propagations * 1e-6, w));
  msg ("decisions %ld (%.1f propagations per decisions)",
       decisions, average (propagations, decisions));
  msg ("conflicts %ld (%.1f decisions per conflict)",
       conflicts, average (decisions, conflicts));
  msg ("learned %d (%.0f%% learned unit clauses per conflict)",
       num_learned, percent (num_learned, conflicts));
  msg ("implied %d (%.1f per learned unit clause)",
       num_implied, average (num_implied, num_learned));
  msg ("");
  msg ("process time %.3f seconds", w);
  msg ("wall clock time %.3f seconds", w);
  msg ("utilization %.0f%% for %d threads",
       percent (p, w * num_threads), num_threads);
}

/*------------------------------------------------------------------------*/

// Probing.

typedef struct Worker Worker;

struct Worker
{
  int id;
  int level;
  int imported;			// imported globally learned
  int completed;		// completed globally learned
  int *trail;			// stack of assigned literals
  int assigned;			// number of literals on trail
  int propagated;		// next literal on trail to propagate
  unsigned *stamps;		// 'num_learned' when last propagated
  signed char *values;		// -1=false, 0=unassigned, 1=true
  pthread_t thread;
  long propagations;
  long decisions;
  long conflicts;
};

#ifdef LOGGING

// In order to enable logging configure with './configure.sh -l'.

static void
logging (Worker * worker, const char *fmt, ...)
{
  if (pthread_mutex_lock (&message_mutex))
    die ("failed to lock messages during logging");
  va_list ap;
  printf ("c [LOGGING] worker[%d] ", worker->id);
  va_start (ap, fmt);
  vprintf (fmt, ap);
  va_end (ap);
  fputc ('\n', stdout);
  fflush (stdout);
  if (pthread_mutex_unlock (&message_mutex))
    die ("failed to unlock messages during logging");
}

#else
#define logging(...) do { } while (0)
#endif

static Worker *workers;
static int num_workers;

static void
init_worker (Worker * worker)
{
  int id = (int) (worker - workers);

  worker->id = id;
  worker->completed = -1;	// Need to propagate all literals first.

  logging (worker, "completed %d", worker->completed);

  worker->trail = malloc (num_variables * sizeof *worker->trail);
  if (num_variables && !worker->trail)
    die ("failed to allocate trail of worker %d", id);

  worker->assigned = worker->propagated = 0;

  worker->values = calloc (2 * (num_variables + 1u), sizeof *worker->values);
  if (!worker->values)
    die ("failed to allocate values of worker %d", id);
  worker->values += num_variables;

  worker->stamps = calloc (2 * (num_variables + 1u), sizeof *worker->stamps);
  if (!worker->stamps)
    die ("failed to allocate stamps of worker %d", id);
  worker->stamps += num_variables;
  num_workers++;

  logging (worker, "initialized");
}

static void
reset_worker (Worker * worker)
{
  logging (worker, "reset");

  propagations += worker->propagations;
  decisions += worker->decisions;
  conflicts += worker->conflicts;

  worker->values -= num_variables;
  free (worker->values);

  worker->stamps -= num_variables;
  free (worker->stamps);

  free (worker->trail);
}

static void
init_workers (void)
{
  workers = calloc (num_threads, sizeof *workers);
  if (!workers)
    die ("failed to allocate workers");
  for (int i = 0; i < num_threads; i++)
    init_worker (workers + i);
  assert (num_workers == num_threads);
}

static void
reset_workers (void)
{
  for (int i = 0; i < num_workers; i++)
    reset_worker (workers + i);
  free (workers);
}

static void
assign (Worker * worker, int lit)
{
  logging (worker, "%s %d", worker->level ? "assign" : "forced", lit);

  assert (!worker->values[lit]);
  assert (!worker->values[-lit]);
  worker->values[lit] = 1;
  worker->values[-lit] = -1;
  assert (worker->assigned < num_variables);

  // Saved assigned literal on 'trail', which acts as a working queue to
  // propagate literals through the 'propagate' offset up to 'assigned'.
  //
  worker->trail[worker->assigned++] = lit;
}

static void
unassign (Worker * worker, int lit)
{
  logging (worker, "unassign %d", lit);

  assert (worker->values[lit] > 0);
  assert (worker->values[-lit] < 0);
  worker->values[lit] = worker->values[-lit] = 0;
}

// This function is used in two context.

static void import_learned (Worker * worker);

static void
new_learned_unit_clause (Worker * worker, int unit)
{
  if (pthread_mutex_lock (&learned_mutex))
    die ("could not lock learned");

  import_learned (worker);

  if (!inconsistent)
    {
      const signed char value = worker->values[unit];
      if (!value)
	{
	  assert (num_learned < num_variables);
	  logging (worker, "learned[%d] %d", num_learned, unit);
	  learned[num_learned++] = unit;
	  assign (worker, unit);
	  worker->imported++;
	}
      else if (value < 0)
	inconsistent = true;
    }

  if (pthread_mutex_unlock (&learned_mutex))
    die ("could not unlock learned");
}

static bool
propagate (Worker * worker)
{
  int saved = worker->propagated;

  while (worker->propagated < worker->assigned)
    {
      const int lit = worker->trail[worker->propagated++];
      assert (worker->values[lit] > 0);

      logging (worker, "propagate %d", lit);

      // Now go through all clauses with '-lit'.
      //
      int *clause;
      for (int **w = watches[-lit]; (clause = *w); w++)
	{
	  int unassigned = 0;
	  int satisfied = 0;
	  int other = 0;
	  int unit = 0;

	  for (const int *p = clause; (other = *p); p++)
	    {
	      const signed char value = worker->values[other];

	      if (value < 0)	// Skip falsified literals.
		continue;

	      if (value > 0)	// This clause is satisfied.
		{
		  satisfied++;
		  break;
		}

	      assert (!value);
	      if (unassigned++)
		break;		// Found two unassigned literals.

	      unit = *p;
	    }

	  if (satisfied)
	    continue;

	  if (unassigned > 1)
	    continue;

	  if (!unassigned)
	    return false;	// Found conflict since all falsified.

	  assert (unassigned == 1);
	  assert (unit);

	  assign (worker, unit);
	}
    }

  worker->propagations += worker->propagated - saved;

  return true;
}

static void
import_learned (Worker * worker)
{
  int imported = 0;

  while (worker->imported < ACCESS (num_learned))
    {
      int other = learned[worker->imported];
      logging (worker, "import[%d] %d", worker->imported, other);
      worker->imported++;

      signed char value = worker->values[other];

      if (value > 0)
	{
	  logging (worker, "satisfied %d", other);
	  continue;
	}

      if (value < 0)
	{
	  logging (worker, "falsified %d", other);
	  inconsistent = true;
	  break;
	}

      assign (worker, other);
      imported++;
    }

  if (imported && !inconsistent)
    {
      assert (!worker->level);
      if (!propagate (worker))
	inconsistent = true;
    }
}

static void
probe_literal (Worker * worker, int probe)
{
  assert (probe);

  import_learned (worker);

  if (worker->values[probe])
    {
      logging (worker, "probe %d already assigned", probe);
      return;
    }

  // Check if since the last time this 'probe' was propagated and did not
  // lead to a conflict a new unit was found. If not, no need to propagate.
  //
  const unsigned stamp = ACCESS (num_learned) + 1u;
  if (worker->stamps[probe] >= stamp)
    {
      logging (worker, "probe %d already tried", probe);
      return;
    }

  logging (worker, "probing %d", probe);

  assert (!worker->level);
  worker->level = 1;

  const int saved = worker->assigned;
  assign (worker, probe);
  worker->decisions++;

  if (propagate (worker))	// Propagation does not lead to a conflict.
    {
      logging (worker, "consistent %d", probe);

      // Backtrack to previous state by unassigning the probing literal
      // 'probe' and all the implied literals.  Mark all those literals as
      // having been tried until the current number of learned unit clauses.

      while (saved < worker->assigned)
	{
	  const int other = worker->trail[--worker->assigned];
	  unassign (worker, other);

	  // Remember that for the number of learned unit clauses found so
	  // far this other literal has been propagated and did not lead to
	  // a conflict.
	  //
	  worker->stamps[other] = stamp;
	}
      worker->propagated = worker->assigned;

      assert (worker->level == 1);
      worker->level = 0;
    }
  else				// Propagation lead to a conflict.
    {
      logging (worker, "failed %d", probe);

      worker->conflicts++;	// Thus 'probe' is a failed literal.

      // First backtrack (without time stamping).
      //
      while (saved < worker->assigned)
	unassign (worker, worker->trail[--worker->assigned]);
      worker->propagated = worker->assigned;

      assert (worker->level == 1);
      worker->level = 0;

      // Save resulting learned unit clause (but not all the implied
      // literals derived in the following propagation).
      //
      const unsigned unit = -probe;
      new_learned_unit_clause (worker, unit);

      // Propagate locally.
      //
      if (!propagate (worker))
	inconsistent = true;	// Mark formula as being unsatisfiable.
    }
}

static void
probe_all_literals (Worker * worker)
{
  logging (worker, "starting");

#ifdef LOGGING
  long round = 0;
#endif

  bool done = false;

  while (!done)
    {
      logging (worker, "begin %ld", ++round);

      if (pthread_mutex_lock (&learned_mutex))
	die ("can not lock to save learned");

      const int saved = num_learned;

      if (pthread_mutex_unlock (&learned_mutex))
	die ("can not unlock after saving learned");

      // First worker works on the first literal, second on the second etc.

      const int start = -num_variables + worker->id;
      const int delta = num_workers;

      for (int lit = start; lit <= num_variables; lit += delta)
	{
	  if (!lit)
	    continue;
	  if (inconsistent)
	    break;
	  probe_literal (worker, lit);
	}

      logging (worker, "end %ld", round);

      if (inconsistent)
	break;

      if (pthread_mutex_lock (&learned_mutex))
	die ("can not lock to check termination");

      if (saved == num_learned)
	{
	  worker->completed = saved;
	  logging (worker, "completed %d", worker->completed);

	  done = true;

	  for (int i = 0; done && i < num_workers; i++)
	    if (workers[i].completed < num_learned)
	      {
		logging (worker, "worker[%d].completed %d < learned %d",
			 i, workers[i].completed, num_learned);
		done = false;
	      }
	}
      else
	{
	  logging (worker, "saved %d < learned %d", saved, num_learned);
	  done = false;
	}

      if (pthread_mutex_unlock (&learned_mutex))
	die ("can not unlock after checking termination");
    }

  logging (worker, "finished");
}

static void
propagate_original_units (Worker * worker)
{
  for (int i = 0; !inconsistent && i < num_units; i++)
    {
      int unit = units[i];
      const signed char value = worker->values[unit];
      if (value > 0)
	continue;
      if (value < 0)
	inconsistent = true;
      if (!value)
	assign (worker, unit);
    }
  if (!propagate (worker))
    inconsistent = true;
}

static void *
run_worker (void *p)
{
  Worker *worker = p;
  propagate_original_units (worker);
  probe_all_literals (worker);
  return 0;
}

static void
run_workers (void)
{
  for (int i = 0; i < num_workers; i++)
    {
      Worker *worker = workers + i;
      if (pthread_create (&worker->thread, 0, run_worker, worker))
	die ("can not create worker thread %d", i);
    }

  msg ("all %d threads created", num_workers);

  for (int i = 0; i < num_workers; i++)
    if (pthread_join (workers[i].thread, 0))
      die ("failed to join worker thread %d", i);

  msg ("all %d threads joined", num_workers);
}

/*------------------------------------------------------------------------*/

// Simplifying.

static void
simplify_clauses (Worker * worker)
{
  int remain = 0;
  for (int i = 0; i < num_clauses; i++)
    {
      int *clause = clauses[i];
      bool satisfied = false;
      int *q = clause, lit;
      for (const int *p = q; !satisfied && (lit = *p); p++)
	{
	  const signed char value = worker->values[lit];
	  if (value > 0)
	    satisfied = true;	// Found true literal.
	  else if (!value)
	    *q++ = lit;		// Only keep unassigned literals.
	}

      if (satisfied)
	free (clause);
      else
	{
	  clauses[remain++] = clause;
	  *q = 0;
	}
    }
  msg ("removed %d clauses (%d remain %.0f%%)",
       num_clauses - remain, remain, percent (remain, num_clauses));
  num_clauses = remain;

  implied = malloc (worker->assigned * sizeof *implied);
  if (worker->assigned && !implied)
    die ("failed to allocate implied literals");

  for (int idx = 1; idx <= num_variables; idx++)
    {
      const signed char value = worker->values[idx];
      if (value)
	implied[num_implied++] = value * idx;
    }
  assert (num_implied == worker->assigned);
}

/*------------------------------------------------------------------------*/

// Printing.

static FILE *output_file;
static const char *output_path;

static void
print (void)
{
  FILE *file = output_file ? output_file : stdout;
  if (inconsistent)
    fprintf (file, "p cnf %d 1\n0\n", num_variables);
  else
    {
      fprintf (file, "p cnf %d %d\n",
	       num_variables, num_implied + num_clauses);
      for (int i = 0; i < num_implied; i++)
	fprintf (file, "%d 0\n", implied[i]);
      for (int i = 0; i < num_clauses; i++)
	{
	  for (const int *p = clauses[i]; *p; p++)
	    fprintf (file, "%d ", *p);
	  fprintf (file, "0\n");
	}
    }
}

/*------------------------------------------------------------------------*/

// Compressed file reading.

static bool
has_suffix (const char *str, const char *suffix)
{
  size_t l = strlen (str), k = strlen (suffix);
  return k <= l && !strcmp (str + l - k, suffix);
}

static void
open_compressed_input (const char *fmt)
{
  assert (input_path);
  char *cmd = malloc (strlen (fmt) + strlen (input_path));
  sprintf (cmd, fmt, input_path);
  input_file = popen (cmd, "r");
  free (cmd);
  if (!input_file)
    die ("can not read compressed '%s'", input_path);
  close_input = 2;
}

/*------------------------------------------------------------------------*/

int
main (int argc, char **argv)
{
  started = wall_clock_time ();
  for (int i = 1; i < argc; i++)
    {
      const char *arg = argv[i];
      if (!strcmp (arg, "-h"))
	{
	  printf ("usage: parfail [-h] [--version] "
		  "[<threads>] [<input> [<output>]]\n");
	  exit (0);
	}
      else if (!strcmp (arg, "--version"))
	printf ("2.0\n"), exit (0);
      else if (arg[0] == '-' && arg[1])
	die ("invalid option '%s' (try '-h')", arg);
      else if (is_number (arg))
	{
	  if (num_threads)
	    die ("multiple thread options '%d' and '%s'", num_threads, arg);
	  if ((num_threads = atoi (arg)) <= 0)
	    die ("invalid number of threads '%s'", arg);
	}
      else if (output_path)
	die ("too many files '%s' and '%s'", input_path, output_path);
      else if (!input_path)
	{
	  if (!strcmp (arg, "-"))
	    die ("invalid input path '-'");
	  input_path = arg;
	}
      else
	output_path = arg;
    }
  if (!input_path)
    input_path = "<stdin>", input_file = stdin;
  else if (has_suffix (input_path, ".gz"))
    open_compressed_input ("gzip -c -d %s");
  else if (has_suffix (input_path, ".bz2"))
    open_compressed_input ("bzip2 -c -d %s");
  else if (has_suffix (input_path, ".xz"))
    open_compressed_input ("xz -c -d %s");
  else if (!(input_file = fopen (input_path, "r")))
    die ("can not read '%s'", input_path);
  else
    close_input = 1;
  msg ("Parfail Failed Literal Preprocessor");
  msg ("");
  msg ("reading DIMACS file from '%s'", input_path);
  parse ();
  if (close_input == 1)
    fclose (input_file);
  if (close_input == 2)
    pclose (input_file);

  init_learned_unit_clauses ();
  init_watches ();
  init_threads ();
  init_workers ();
  run_workers ();

  reset_watches ();
  if (!inconsistent)
    simplify_clauses (workers);
  reset_workers ();

  if (inconsistent)
    msg ("formula UNSATISFIABLE (inconsistent)");
  else if (num_implied == num_variables)
    msg ("formula SATISFIABLE (all variables assigned)");
  else
    msg ("formula not solved");

  msg ("");

  if (output_path && !strcmp (output_path, "-"))
    {
      msg ("writing simplified formula to '<stdout>'");
      print ();
    }
  else if (!output_path)
    msg ("not writing simplified CNF (use '-' to write to '<stdout>')");
  else if (!(output_file = fopen (output_path, "w")))
    die ("can not write to '%s'", output_path);
  else
    {
      msg ("writing simplified formula to '%s'", output_path);
      print ();
      fclose (output_file);
    }
  reset_clauses ();
  free (learned);
  free (implied);
  statistics ();
  return 0;
}
