#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
#include <assert.h>
#include <pthread.h>
#include "exerr.h"

/* 'LockedElem' consists of an element and a lock protecting the element. */
struct LockedElem
{
  pthread_mutex_t elem_mutex;
  unsigned int elem;
};

typedef struct LockedElem LockedElem;


#define ARRAY_SIZE (1 << 4)
#define NUM_THREADS (1 << 3)
#define THREAD_ROUNDS (1)
#define THREAD_WORKTIME (2)
#define GET_RAND_VALUE(max) (rand() % (max))
static int seed = 0;
/* Threads will access one element of 'array' and simulate some work. */
static LockedElem array[ARRAY_SIZE];
static pthread_t thread_ids[NUM_THREADS];


static void
init_array ()
{
  int r;
  unsigned int i;
  for (i = 0; i < ARRAY_SIZE; i++)
    {
      array[i].elem = 0;
      pthread_mutex_t *mutex = &(array[i].elem_mutex); 
      r = pthread_mutex_init (mutex, NULL);
      COND_MSG_ABORT (r, r, "Mutex init");
    }
}


static void *
thread_func (void *arg)
{
  unsigned long int uint_id = (unsigned long int) arg;
  unsigned int i, rounds;
  int r;
  rounds = THREAD_ROUNDS;
  for (i = 0; i < rounds; i++)
    {
      /* Pick position to be accessed at random. */
      unsigned int pos =  GET_RAND_VALUE(ARRAY_SIZE);
      /* 'worktime' simulates some expensive work by putting thread to sleep. */
      unsigned int worktime =  THREAD_WORKTIME;
      /* A thread will lock the element to be accessed. This does not
	 prevent other threads from accessing different elements. In the
	 best case, this allows full parallelism unless two threads want
	 to access the same element. */
      r = pthread_mutex_lock(&(array[pos].elem_mutex));
      COND_MSG_ABORT (r, r, "Mutex lock");
      /* Do some work. */
      fprintf (stderr, "Thread %lu writes pos %u and works for %u seconds.\n", uint_id, pos, worktime);
      array[pos].elem++;
      sleep (worktime);
      fprintf (stderr, "\tThread %lu releases lock of pos %u.\n", uint_id, pos);
      r = pthread_mutex_unlock(&(array[pos].elem_mutex));
      COND_MSG_ABORT (r, r, "Mutex unlock");
    }
  pthread_exit (NULL);
}


int
main ()
{
  int r;
  unsigned long int i;

  init_array();
  srand (seed);

  /* Create 'NUM_THREADS' threads which access positions of array randomly. */
  for (i = 0; i < NUM_THREADS; i++)
    {
      r = pthread_create (thread_ids + i, NULL, thread_func, (void*)i);
      COND_MSG_ABORT (r, r, "Thread creation");
    }

  /* Wait for threads to finish. */
  for (i = 0; i < NUM_THREADS; i++)
    {
      r = pthread_join (*(thread_ids + i), NULL);
      COND_MSG_ABORT (r, r, "Thread join");
    }

  pthread_exit (NULL);
}
