#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include <sys/time.h>
#include <sys/resource.h>

#include <pthread.h>

long l;

pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
int active;
int w;

long sequential_fib (long n) {
  if (n < 2) return n;
  long x = sequential_fib (n - 1);
  long y = sequential_fib (n - 2);
  return x + y;
}

long parallel_fib (long n) {
  if (n < l) return sequential_fib (n);
  pthread_t thread;
  pthread_mutex_lock (&mutex);
  int spawn = w > active;
  if (spawn) active++;
  pthread_mutex_unlock (&mutex);
  long x, y;
  if (spawn) {
    pthread_create (&thread, 0,
      (void*(*)(void*)) parallel_fib, (void *)(n - 1));
    y = parallel_fib (n - 2);
    pthread_join (thread, (void**) &x);
    pthread_mutex_lock (&mutex);
    assert (active > 0);
    active--;
    pthread_mutex_unlock (&mutex);
  } else {
    x = parallel_fib (n - 1);
    y = parallel_fib (n - 2);
  }
  long r = x + y;
  return r;
}

double wall_clock_time () {
  double res = 0;
  struct timeval tv;
  if (!gettimeofday (&tv, 0))
  res = 1e-6 * tv.tv_usec, res += tv.tv_sec;
  return res;
}

double process_time () {
  struct rusage u;
  if (getrusage (RUSAGE_SELF, &u)) return 0;
  double res = u.ru_utime.tv_sec + 1e-6 * u.ru_utime.tv_usec;
  res += u.ru_stime.tv_sec + 1e-6 * u.ru_stime.tv_usec;
  return res;
}

int main (int argc, char ** argv) {
  if (argc < 2 || argc > 4) {
    printf (
"usage: pfib <n> [ <w> [ <l> ] ]\n"
"\n"
"Computes fibonacci number <n>\n"
"using <w> worker threads (default 1) and\n"
"leaf coarsening limit <l> (default 2).\n");
    exit (0);
  }
  long n = atol (argv[1]);
  const char * warg = argc > 2 ? argv[2] : "1";
  if (argc > 3) l = atoi (argv[3]);
  if (l < 2) l = 2;
  w = atoi (warg);
  if (w < 1) w = 1;
  double s = wall_clock_time ();
  long f = parallel_fib (n);
  double t = wall_clock_time () - s;
  double p = process_time ();
  double u = (t && w) ? 100.0 * p / t / (double) w : 0;
  printf (
    "fib (%ld) = %ld with %d threads leaf limit %ld in %.2f seconds\n"
    "%.2f process time, utilization %.0f%%\n",
    n, f, w, l, t, p, u);
  return 0;
}
