#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <pthread.h>
#include <assert.h>
#include <time.h>
#include "exerr.h"

/* This is a clean solution of example 'returnval_heap' using condition
   variables for thread notification. */

#define NUM_THREADS 20
#define MAX_SLEEP_TIME 7

/* if enabled, then main-thread will periodically timeout while waiting */
#define TIMEDWAIT 0

#if TIMEDWAIT
#define TIMEOUT 2
#endif

static pthread_t threads[NUM_THREADS];
static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t cond = PTHREAD_COND_INITIALIZER;

/* number of threads which have placed return value on heap;
   main-thread reads return values if 'finished == NUM_THREADS' */
static int finished = 0;


typedef struct Data
{
  int val;
} Data;

typedef struct Result
{
  int id;
  int int_res;
} Result;

/* threads will index array by assigned, unique IDs */
static Result *results[NUM_THREADS];


void *
heap_return (void *arg)
{
  int r;

  /* all resources may be released after termination */
  r = pthread_detach (pthread_self ());
  COND_MSG_ABORT (r, r, "Thread detach");

  sleep (rand () % (MAX_SLEEP_TIME + 1));

  Data *data = (Data *) arg;
  int id = data->val;
  free (data);

  /* thread stores return value on the heap */
  Result *res = (Result *) malloc (sizeof (Result));
  res->id = id;
  res->int_res = id * 100;

  results[id] = res;

  r = pthread_mutex_lock (&mutex);
  COND_MSG_ABORT (r, r, "Mutex lock");

  finished++;
  printf ("Thread %d has written result (totally %d finished)\n", id,
	  finished);
  if (finished == NUM_THREADS)
    {
      /* need not broadcast since there is only main-thread waiting */
      r = pthread_cond_signal (&cond);
      COND_MSG_ABORT (r, r, "Cond signal");
    }
  r = pthread_mutex_unlock (&mutex);
  COND_MSG_ABORT (r, r, "Mutex unlock");

  pthread_exit (NULL);
}


int
main (int argc, char **argv)
{
  srand (time (NULL));

  int i, r;
  for (i = 0; i < NUM_THREADS; i++)
    {
      Data *data = (Data *) malloc (sizeof (Data));
      data->val = i;
      printf ("Creating thread %d\n", i);
      r = pthread_create (&threads[i], NULL, heap_return, (void *) data);
      COND_MSG_ABORT (r, r, "Thread creation");
    }

#if TIMEDWAIT
  struct timespec timeout;
  timeout.tv_sec = time (NULL) + TIMEOUT;	/* absolute timeout */
  timeout.tv_nsec = 0;
#endif

  r = pthread_mutex_lock (&mutex);
  COND_MSG_ABORT (r, r, "Mutex lock");

  while (finished < NUM_THREADS)
    {				/* main-thread is notified exactly once if 'finished == NUM_THREADS' */
#if !TIMEDWAIT
      r = pthread_cond_wait (&cond, &mutex);
      COND_MSG_ABORT (r, r, "Cond wait");
#else
      r = pthread_cond_timedwait (&cond, &mutex, &timeout);
      if (r == ETIMEDOUT)
	{
	  printf ("Main-thread timed out\n");
	  timeout.tv_sec = time (NULL) + TIMEOUT;	/* set new timeout */
	  timeout.tv_nsec = 0;
	}
      else 
	COND_MSG_ABORT (r, r, "Cond timedwait");
#endif
    }
  r = pthread_mutex_unlock (&mutex);
  COND_MSG_ABORT (r, r, "Mutex unlock");
  printf ("Main-thread has been notified\n");

  Result *resp;
  for (i = 0; i < NUM_THREADS; i++)
    {
      resp = results[i];
      assert (i == results[i]->id);
      printf ("Thread %d returned %d\n", i, resp->int_res);
      free (results[i]);
    }

  pthread_exit (NULL);
}
